/******************************************************************************
 *   fsc-celsius-m470-sel - Small userspace tool to read to contents of the
 *                 system eventlog on Fujitsu Siemens Celsius m470 computers
 *
 *   Copyright (C) 2010 Mathias Kettner GmbH,
 *                      Lars Michelsen <lm@mathias-kettner.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *   MA 02110-1301 USA.
 *****************************************************************************/

/** 
 * This program has been developed to have a small code to read the contents
 * of the DMI/SMBIOS eventlog in the BIOS. It has been built on the base of
 * the SMBIOS 2.x specifications which can be found here:
 *   http://www.phoenix.com/docs/specssmbios.pdf
 *
 * The intention is to keep this program as simple as possible so it does not
 * search for the location of the SEL itselfs. It takes the memory address by
 * parameter. The correnct memory address can be found using dmidecode:
 * "dmidecode --type 15".
 */

#include <fcntl.h>
#include <sys/mman.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <limits.h>

#define PROG                  "fsc-celsius-m470-sel"
#define VERSION               "1.2"

#define SEL_FILE              "/dev/mem"
#define SEL_HEAD              0x0000
#define SEL_HEAD_LEN          16
#define SEL_DATA_OFFSET       0x0010

#define SEL_HEAD_OEM_RES0     0x00
#define SEL_HEAD_OEM_RES1     0x01
#define SEL_HEAD_OEM_RES2     0x02
#define SEL_HEAD_OEM_RES3     0x03
#define SEL_HEAD_OEM_RES4     0x04
#define SEL_HEAD_ME_TIME      0x05
#define SEL_HEAD_ME_COUNT     0x06
#define SEL_HEAD_PB_RES_ADDR  0x07
#define SEL_HEAD_PB_RES_INDEX 0x08
#define SEL_HEAD_CM_CS_START  0x09
#define SEL_HEAD_CM_CS_COUNT  0x0a
#define SEL_HEAD_CM_CS_OFFSET 0x0b
#define SEL_HEAD_RES0         0x0c
#define SEL_HEAD_RES1         0x0d
#define SEL_HEAD_RES2         0x0e
#define SEL_HEAD_REV          0x0f

#define SEL_LOG_TYPE          0x00
#define SEL_LOG_LEN           0x01
#define SEL_LOG_YEAR          0x02
#define SEL_LOG_MONTH         0x03
#define SEL_LOG_DAY           0x04
#define SEL_LOG_HOUR          0x05
#define SEL_LOG_MIN           0x06
#define SEL_LOG_SEC           0x07
#define SEL_LOG_VAR           0x08

struct message {
  int val;
  const char * label;
};

struct message messages[] = {
	{0x00,  "Unknown message type" },
	{0x01,  "Single-bit ECC memory error" },
	{0x02,  "Multi-bit ECC memory error" },
	{0x03,  "Parity memory error" },
	{0x04,  "Bus time-out" },
	{0x05,  "I/O Channel Check" },
	{0x06,  "Software NMI" },
	{0x07,  "POST Memory Resize" },
	{0x08,  "POST Error" },
	{0x09,  "PCI Parity Error" },
	{0x0A, "PCI System Error" },
	{0x0B, "CPU Failure" },
	{0x0C, "EISA FailSafe Timer time-out" },
	{0x0D, "Correctable memory log disabled" },
	{0x0E, "Logging disabled for a specific Event Type - too many errors of the same type received in a short amount of time" },
	{0x0F, "Unknown message type" },
	{0x10, "System Limit Exceeded (e.g. voltage or temperature threshold)" },
	{0x11, "Asynchronous hardware timer expired and issued a system reset" },
	{0x12, "System configuration information" },
	{0x13, "Hard-disk information" },
	{0x14, "System reconfigured" },
	{0x15, "Uncorrectable CPU-complex error" },
	{0x16, "Log Area Reset/Cleared" },
	{0x17, "System boot" }
        // 0x18-0x7F  Unused, available for assignment by this specification.
        // 0x80-0xFE   Available for system- and OEM-specific assignments.
        // 0xFF        End-of-Log
};

static void help(void) {
  fprintf(stderr,
    "Usage: %s ADDRESS \n"
    "  ADDRESS is the address of the SEL like reported by dmidecode\n"
    "\n", PROG);
}

int main(int argc, char *argv[])
{
  int flags = 0, version = 0, debug = 0, sel_offset, sel_page, sel_len;
  
  int page_size = sysconf(_SC_PAGE_SIZE);
  int mem_read = page_size * 16;
  mem_read = 65536;
  
  while (1+flags < argc && argv[1+flags][0] == '-') {
    switch (argv[1+flags][1]) {
    case 'V': version = 1; break;
    case 'd': debug = 1; break;
    case 'h':
      help();
      exit(1);
    default:
      printf("E 0000-00-00 00:00:00 Error: Unsupported option "
                                        "\"%s\"!\n", argv[1+flags]);
    }
    flags++;
  }
  
  if(version) {
    printf("%s version %s\n", PROG, VERSION);
    exit(0);
  }
  
  if (argc < flags + 3) {
    printf("E 0000-00-00 00:00:00 Error: No address or length specified!\n");
    exit(1);
  } else {
    // Read address
    const char * arg_addr = argv[flags+1];
    if(arg_addr[0] == '0' && arg_addr[1] == 'x') {
      // Get and convert address to int
      long sel_addr   = strtoul(arg_addr, 0, 0);
      if(errno == ERANGE || sel_addr == ULONG_MAX) {
        printf("E 0000-00-00 00:00:00 Unable to transform address: %s\n",
                                                            strerror(errno));
        exit(1);
      }
      
      sel_page       = sel_addr & ~ 0xffff;
      sel_offset     = sel_addr & 0xffff;
    } else {
      printf("E 0000-00-00 00:00:00 Error: Wrong address format!\n");
      exit(1);
    }

    // Read area length
    const char * arg_len = argv[flags+2];
    // Get and convert length into int
    sel_len = strtoul(arg_len, 0, 0);
    if(errno == ERANGE || sel_len == ULONG_MAX) {
      printf("E 0000-00-00 00:00:00 Unable to transform address: %s\n",
                                                          strerror(errno));
      exit(1);
    }

  }

  if(debug == 1) {
    printf("SEL Length:  %d\n", sel_len);
    printf("SEL Page:    0x%02x\n", sel_page);
    printf("SEL Offset:  0x%02x\n", sel_offset);
    printf("Page Size:   %d\n", page_size);
  }

  int fd = open(SEL_FILE, O_RDONLY);
  if(fd == -1) {
    printf("E 0000-00-00 00:00:00 Failed opening %s: %s\n",
                                      SEL_FILE, strerror(errno));
    close(fd);
  }

  if(debug == 1) {
    printf("Reading %d from 0x%02x\n", mem_read, sel_page);
  }
  
  char *sel = mmap(NULL, mem_read, PROT_READ, MAP_SHARED, fd, sel_page);
  if(sel == MAP_FAILED) {
    printf("E 0000-00-00 00:00:00 Failed reading from %s: %s\n",
                                        SEL_FILE, strerror(errno));
    close(fd);
    exit(1);
  }

  // sel+sel_offset points to SEL "Access Method Address"
  // like described in "3.3.16.5.1 Log Header Type 1 Format"
  // in SMBIOS Reference Specification Version 2.3
  //
  // This dumps the SEL header (16 bytes)
  if(debug == 1) {
    int a;
    for(a = 0; a <= 16; a++) {
      printf("%02x 0x%02x\n", a, sel[a+sel_offset]);
    }
  }

  // Perform some validations using the SEL header:
  // 00 0x00 OEM Reserved
  // 01 0x00 OEM Reserved
  // 02 0x00 OEM Reserved
  // 03 0x00 OEM Reserved
  // 04 0x00 OEM Reserved
  // 05 0x60 Multiple Event Time Window (Minutes in BCD)
  // 06 0x01 Multiple Event Count Increment (Number of Events
  //         to pass before the counter gets increased.
  //         1-255 is valid)
  // 07 0x7c Pre-Boot Event Log Reset - CMOS Address
  // 08 0x00 Pre-boot Event Log Reset — CMOS Bit Index
  // 09 0x00 CMOS Checksum - Starting Offset
  // 0a 0x01 CMOS Checksum - Byte Count
  // 0b 0x7d CMOS Checksum - Checksum Offset
  // 0c 0x00 Reserved
  // 0d 0x00 Reserved
  // 0e 0x00 Reserved
  // 0f 0x01 Header Revision
  int valid = 1;
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_OEM_RES0] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x00 is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_OEM_RES1] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x01 is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_OEM_RES2] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x02 is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_OEM_RES3] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x03 is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_OEM_RES4] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x04 is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_RES0] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x0c is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_RES1] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x0d is not 0x00\n");
    valid = 0;
  }
  if(sel[sel_offset+SEL_HEAD+SEL_HEAD_RES2] != 0x00) {
    printf("E 0000-00-00 00:00:00 SEL Validation failed: "
           "Header byte 0x0e is not 0x00\n");
    valid = 0;
  }
  
  // Terminate on validation problems
  if(valid != 1) {
    munmap(sel, mem_read);
    close(fd);
    exit(1);
  }

  int i, type, len = 0, year;
  char fullyear[4], month[3], day[3];
  char hour[3], min[3], sec[3];

  // Set address to start of the SEL Data area
  int addr = sel_offset + SEL_DATA_OFFSET;

  // Catch empty logs
  if(sel[addr] == -1) {
    printf("I 0000-00-00 00:00:00 Found no log entries.\n");
    munmap(sel, mem_read);
    close(fd);
    exit(1);
  }

  // Loop the log until type is -1 (Delimiter)
  do {
    type = sel[addr+SEL_LOG_TYPE];

    // 0x18-0x7F   Unused, available for assignment by this specification.
    // 0x80-0xFE   Available for system- and OEM-specific assignments.
    // 0xFF        End-of-Log

    const char * text;
    if(type > sizeof(messages) / sizeof(struct message)) {
      if(type < 0xFF) {
        text = "Unhandled message type";
      } else {
        printf("E 0000-00-00 00:00:00 Invalid message type (%d / %0x02x) found\n", type, type);
        munmap(sel, mem_read);
        close(fd);
        exit(1);
      }
    } else {
      text = messages[type].label;
    }

    len = 128 + sel[addr+SEL_LOG_LEN];
  
    // This code dumps the current log message
    if(debug == 1) {
      printf("Dumping log message:\n");
      int a;
      for(a = 0; a <= len; a++) {
        printf("%02x 0x%02x\n", a, sel[addr+a]);
      }
    }
    
    // One log message:
    // 00 0x01         Event Type
    // 01 0xffffff8c   Length of Log
    // 02 0x10         Year
    // 03 0x03         Month
    // 04 0x29         Day
    // 05 0x07         Hour
    // 06 0x25         Minute
    // 07 0x48         Second
    // 08 0x00         Log variable Data ...
    // 09 0x00         ...
    // 0a 0x00         ...
    // 0b 0x00         ...
    // 0c 0xffffffff   Log Delim

    int l = 0;

    // Note: The century portion of the two-digit year is implied as
    // ‘19’ for year values in the range 80h to 99h and ‘20’ for year
    // values in the range 00h to 79h.
    year = sel[addr+SEL_LOG_YEAR];
    if(year <= 0x79) {
      l = sprintf(fullyear, "20%02x", year);
    } else {
      l = sprintf(fullyear, "19%02x", year);
    }

    snprintf(month, 3, "%02x", sel[addr+SEL_LOG_MONTH]);
    snprintf(day,   3, "%02x", sel[addr+SEL_LOG_DAY]);
    snprintf(hour,  3, "%02x", sel[addr+SEL_LOG_HOUR]);
    snprintf(min,   3, "%02x", sel[addr+SEL_LOG_MIN]);
    snprintf(sec,   3, "%02x", sel[addr+SEL_LOG_SEC]);

    if(debug == 1) {
      printf("New Log:");
      printf("  Type: %d Len: %d\n", type, len);

      printf("  Date: %s-%s-%s %s:%s:%s\n", fullyear, month, day, hour, min, sec);
      printf("  Message: %s\n", text);
    }

    printf("%d %s-%s-%s %s:%s:%s %s\n",
      type, fullyear, month, day, hour, min, sec, text);

    addr += len;
  } while(sel[addr] != -1 && addr < addr+sel_len);
  // The second check has been added to prevent long looping problems when
  // the eventlog structure is corrupted or not built as expected. Only looping
  // number of bytes like set by command line parameter and reported by dmidecode.

  munmap(sel, mem_read);
  close(fd);
  exit(0);
}
